/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCWIFIDISPLAYRTP_H_73734014_0db8_11e4_869c_7071bc8f94f5
#define VNCWIFIDISPLAYRTP_H_73734014_0db8_11e4_869c_7071bc8f94f5

/**
 * \file vncwifidisplayrtp.h
 *
 * This file defines data types used by the VNC WiFi Display SDK's internal RTP
 * client.
 *
 * \see VNCWiFiDisplayDecoderOnRTPPacket, VNCWiFiDisplayDecoderRecycleRTPPacket
 */

#include "vncwifidisplaytypes.h"

/**
 * \brief An RTP packet received by the VNC WiFi Display SDK's internal RTP
 * client.
 */
typedef struct VNCWiFiDisplayRTPPacket VNCWiFiDisplayRTPPacket;

/**
 * \brief A struct describing an RTP packet received by the VNC WiFi Display
 * SDK's internal RTP client.
 *
 * The contents of this struct should not be modified in any way.
 */
struct VNCWiFiDisplayRTPPacket {

  /**
   * \brief A pointer to the RTP payload data.
   */
  vnc_uint8_t* payloadData;

  /**
   * \brief The size of the data pointed to by payloadData.
   *
   * Unit: bytes.
   */
  vnc_size_t payloadLength;

  /**
   * \brief The value of the local system timestamp at the time when the packet
   * was received.
   *
   * Unit: milliseconds.
   */
  vnc_uint64_t localTimestamp;

  /**
   * \brief The remote timestamp at the time the payload data was sampled.
   *
   * In the raw RTP packet, this is based on a 90kHz timer, and may wrap around
   * in the case of overflow.
   *
   * The value provided in remoteTimestampExtended has been extended to 64
   * bits, taking into account wraparound, meaning that this timestamp
   * can be treated as monotonic.
   *
   * The first packet provided to VNCWiFiDisplayDecoderOnRTPPacket() will have
   * an extended timestamp of zero. This value may be negative if subsequent
   * packets have a remote timestamp lower than that of the first packet.
   *
   * Unit: Number of 90kHz intervals.
   */
  vnc_int64_t remoteTimestampExtended;

  /**
   * \brief The remote timestamp at the time the payload data was sampled.
   *
   * This is the raw value as it appeared in the RTP packet. Unlike the value
   * in remoteTimestampExtended, this value is based on a 90kHz timer and may
   * wrap around in the case of overflow.
   *
   * Unit: Number of 90kHz intervals.
   */
  vnc_uint32_t remoteTimestampRaw;

  /**
   * \brief The extended RTP sequence number of the packet.
   *
   * In the raw RTP packet, the sequence number is only 16 bits long, and may
   * wrap around in the case of overflow.
   *
   * The value provided in sequenceNumberExtended has been extended to 64 bits,
   * taking into account wraparound, meaning that this sequence number can be
   * treated as monotonic.
   *
   * The first received packet will have an extended sequence number equal to
   * its 16 bit raw sequence number.
   *
   * This value may be negative if subsequent packets have a sequence number
   * lower than that of the first packet.
   */
  vnc_int64_t sequenceNumberExtended;

  /**
   * \brief The raw RTP sequence number of the packet.
   *
   * This is the raw value as it appeared in the RTP packet. Unlike the value
   * in sequenceNumberExtended, this value will wrap around in the case of
   * overflow.
   */
  vnc_uint16_t sequenceNumberRaw;

  /**
   * \brief The synchronization source identifier from the RTP header.
   *
   * It is not anticipated that a decoder will make use of this field.
   */
  vnc_uint32_t ssrc;

  /**
   * \brief The marker bit from the RTP header.
   *
   * When the payload type is an MPEG2 transport stream, this bit is set
   * whenever the timestamp is discontinuous with respect to the last packet.
   *
   * It is not recommended that this field be relied upon, since it is possible
   * that, over a lossy WiFi connection, the RTP packet containing the marker
   * bit could be dropped.
   */
  vnc_bool_t markerBit;

  /**
   * \brief The payload type from the RTP header.
   *
   * RTP packets containing MPEG2-TS data should have a payload type of 33.
   *
   * It is not anticipated that a decoder should make use of this field.
   */
  vnc_uint8_t payloadType;

  /**
   * \brief A pointer to the raw RTP packet data, including the header.
   *
   * It is not anticipated that a typical decoder plugin will make use of this
   * field. It is included primarily to facilitate the recycling of buffers
   * internally within the SDK, when the
   * VNCWiFiDisplayDecoderRecycleRTPPacket() supporting API is called. Decoders
   * may safely ignore this field.
   */
  vnc_uint8_t* rawRTPData;

  /**
   * \brief The size of the data pointed to by rawRTPData.
   *
   * Unit: bytes.
   */
  vnc_size_t rawRTPDataLength;

  /**
   * \brief The capacity of the buffer pointed to by rawRTPData.
   *
   * Unit: bytes.
   */
  vnc_size_t rawRTPDataCapacity;

  /**
   * \brief The previous RTP packet.
   *
   * This field may be used by decoder plugins to create a linked list of RTP
   * packets, for instance when performing internal queueing.
   */
  VNCWiFiDisplayRTPPacket *prevPacket;

  /**
   * \brief The next RTP packet.
   *
   * This field may be used by decoder plugins to create a linked list of RTP
   * packets, for instance when performing internal queueing.
   */
  VNCWiFiDisplayRTPPacket *nextPacket;

};


#endif // VNCWIFIDISPLAYRTP_H_73734014_0db8_11e4_869c_7071bc8f94f5

